<?php
namespace SocialFeedsPro;

if(!defined('ABSPATH')){
	exit;
}

class InstagramSettings{

	static function validate_token($access_token){

		if(empty($access_token)){
			wp_send_json_error(__('Access token is required', 'socialfeeds-pro'));
		}
		
		$api_base = 'https://graph.instagram.com/v18.0';

		$url = add_query_arg([
			'fields' => 'id,username,name,biography,profile_picture_url,website,followers_count,media_count,account_type',
			'access_token' => $access_token,
		], $api_base . '/me');

		$response = wp_remote_get($url, ['timeout' => '30', 'sslverify' => true]);

		if ( is_wp_error( $response ) ) {
   		/* translators: %s: error message returned by WordPress HTTP request */
			wp_send_json_error( sprintf( __( 'Request failed: %s', 'socialfeeds-pro' ), $response->get_error_message() ) );
		}

		$http_code = wp_remote_retrieve_response_code($response);
		$body = wp_remote_retrieve_body($response);
		$data = json_decode($body, true);

		if(!is_array($data)){
			wp_send_json_error(__('Invalid response received from Instagram.', 'socialfeeds-pro'));
		}

		if(!empty($data['error'])){
			wp_send_json_error(isset($data['error']['message']) ? $data['error']['message'] : __('Instagram returned an unknown error.', 'socialfeeds-pro'));
		}

		if ( $http_code !== 200 ) {
			/* translators: %d: HTTP response code returned by the Instagram API */
			wp_send_json_error( sprintf( __( 'Instagram API returned HTTP %d.', 'socialfeeds-pro' ), $http_code ) );
		}

		if(!isset($data['id']) || !isset($data['username'])){
			wp_send_json_error(__('Invalid Instagram account data.', 'socialfeeds-pro'));
		}

		return [
			'id' => $data['id'],
			'username' => isset($data['username']) ? $data['username'] : '',
			'name' => isset($data['name']) ? $data['name'] : '',
			'biography' => isset($data['biography']) ? $data['biography'] : '',
			'profile_picture_url' => isset($data['profile_picture_url']) ? $data['profile_picture_url'] : '',
			'website' => isset($data['website']) ? $data['website'] : '',
			'followers_count' => isset($data['followers_count']) ? $data['followers_count'] : 0,
			'media_count' => isset($data['media_count']) ? $data['media_count'] : 0,
			'account_type' => isset($data['account_type']) ? $data['account_type'] : 'PERSONAL',
		];
	}

	static function fetch_posts($access_token, $feed_type, $user_id, $limit = 12){
		$data = \SocialFeedsPro\Util::fetch_feed_data($access_token, $limit);
		return isset($data['data']) ? $data['data'] : [];
	}

	static function get_account_id($token, $username_or_id, $api){

		if(ctype_digit($username_or_id)){
			return $username_or_id;
		}

		$url = add_query_arg([
			'user_id' => 'me',
			'fields'  => 'id,username',
			'search_string' => $username_or_id,
			'access_token'  => $token,
		], "$api/ig_users_search");

		$data = self::make_api_request($url);
		return isset($data[0]['id']) ? $data[0]['id'] : false;
	}

	static function make_api_request($url, $max_items = 0){

		$response = wp_remote_get($url, [
			'timeout'   => 30,
			'sslverify'=> true,
		]);

		if(is_wp_error($response)){
			return [];
		}

		$data = json_decode(wp_remote_retrieve_body($response), true);

		if(empty($data) || ! empty($data['error'])){
			return [];
		}

		if(isset($data['data'])){
			return $max_items > 0 ? array_slice($data['data'], 0, $max_items) : $data['data'];
		}

		return [];
	}

	static function clear_cache(){
		$instagram_opts = get_option('socialfeeds_instagram_option', []);
		$feeds = isset($instagram_opts['instagram_feeds']) ? $instagram_opts['instagram_feeds'] : [];
		
		$tokens = [];
		$limits_config = [];

		$g_limit_d = isset($instagram_opts['instagram_number_posts_desktop']) ? intval($instagram_opts['instagram_number_posts_desktop']) : 6;
		$g_limit_m = isset($instagram_opts['instagram_number_posts_mobile']) ? intval($instagram_opts['instagram_number_posts_mobile']) : 6;
		
		// Global token
		if(!empty($instagram_opts['instagram_access_token'])){
			$token = $instagram_opts['instagram_access_token'];
			$tokens[] = $token;
			$limits_config[] = ['token' => $token, 'limit' => $g_limit_d];
			$limits_config[] = ['token' => $token, 'limit' => $g_limit_m];
		}

		foreach($feeds as $feed){
			// Token resolution
			$token = isset($feed['account_token']) ? $feed['account_token'] : (isset($instagram_opts['instagram_access_token']) ? $instagram_opts['instagram_access_token'] : '');
			
			if($token){
				$tokens[] = $token;
				
				$settings = isset($feed['settings']) ? $feed['settings'] : [];
				// Merge global with feed specific
				$merged = array_merge($instagram_opts, $settings);
				
				$limit_d = isset($merged['instagram_number_posts_desktop']) ? intval($merged['instagram_number_posts_desktop']) : 6;
				$limit_m = isset($merged['instagram_number_posts_mobile']) ? intval($merged['instagram_number_posts_mobile']) : 6;
				
				$limits_config[] = ['token' => $token, 'limit' => $limit_d];
				$limits_config[] = ['token' => $token, 'limit' => $limit_m];
			}
		}
		
		// Clear account info caches
		$tokens = array_unique($tokens);
		foreach($tokens as $t){
			delete_transient('socialfeeds_ig_account_' . md5($t));
		}
		
		// Clear feed caches by limit/token
		$serialized = array_map('serialize', $limits_config);
		$unique = array_unique($serialized);
		$limits_config = array_map('unserialize', $unique);
		
		foreach($limits_config as $conf){
			delete_transient('socialfeeds_ig_feed_' . md5($conf['token'] . $conf['limit']));
		}

	}
}
